﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics.CodeAnalysis;
    using System.Threading.Tasks;
    using DocumentFormat.OpenXml.Spreadsheet;
    using Domain.Configurations;
    using Domain.Helpers;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Domain.Entities;
    using Hims.Domain.Repositories.UnitOfWork;
    using Hims.Infrastructure.Configurations;
    using Hims.Shared.UserModels.Queue;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Models.User;
    using Senders;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    using static System.String;

    // ReSharper disable StyleCop.SA1126

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/users")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class UsersController : BaseController
    {
        /// <summary>
        /// The user services.
        /// </summary>
        private readonly IUserService usersServices;

        /// <summary>
        /// The AES helper.
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The account services.
        /// </summary>
        private readonly IAccountService accountServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The email sender.
        /// </summary>
        private readonly IEmailSender emailSender;

        /// <summary>
        /// The sms sender.
        /// </summary>
        private readonly ISMSSender smsSender;

        /// <summary>
        /// The application configuration.
        /// </summary>
        private readonly IApplicationConfiguration applicationConfiguration;


        /// <summary>
        /// The ftp helper.
        /// </summary>
        private readonly IFtpHelper ftpHelper;

        /// <summary>
        /// The FTP upload helper.
        /// </summary>
        private readonly IFtpUploadHelper ftpUploadHelper;

        /// <summary>
        /// The running environment
        /// </summary>
        private readonly IRunningEnvironment runningEnvironment;

        /// <inheritdoc />
        public UsersController(IUserService usersServices,
            IAuditLogService auditLogServices,
            IApplicationConfiguration applicationConfiguration,
            IEmailSender emailSender, IAESHelper aesHelper,
            IDocumentHelper documentHelper,
            IAccountService accountServices,
            ISMSSender smsSender,
            IURLShortnerHelper uRLShortnerHelper,
            
            INewFtpConfiguration NewftpConfiguration,
            IFtpHelper ftpHelper,
            IFtpUploadHelper ftpUploadHelper,
            IRunningEnvironment runningEnvironment
            )
        {
            this.usersServices = usersServices;
            this.auditLogServices = auditLogServices;
            this.applicationConfiguration = applicationConfiguration;
            this.emailSender = emailSender;
            this.aesHelper = aesHelper;
            this.accountServices = accountServices;
            this.smsSender = smsSender;
            this.ftpHelper = ftpHelper;
            this.ftpUploadHelper = ftpUploadHelper;
            this.runningEnvironment = runningEnvironment;
        }

        /// <summary>
        /// The fetch users.
        /// </summary>
        /// <param name="model">
        /// The user filter model.
        /// </param>
        /// <returns>
        /// The list of users.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of users.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [AllowAnonymous]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<UserModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] UserFilterModel model)
        {
            model = (UserFilterModel)EmptyFilter.Handler(model);
            var users = await this.usersServices.FetchAsync(model);
            if (users == null)
            {
                return this.ServerError();
            }

            foreach (var user in users)
            {
                user.EncryptedUserId = this.aesHelper.Encode(user.UserId.ToString());
            }
            return this.Success(users);
        }

        /// <summary>
        /// The find user.
        /// </summary>
        /// <param name="model">
        /// The user filter model.
        /// </param>
        /// <returns>
        /// The user model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - The user model.
        /// - 400 - Sorry! We don't have a user in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("find")]
        [ProducesResponseType(typeof(UserModel), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindAsync([FromBody] FindUserRequest model)
        {
            model = (FindUserRequest)EmptyFilter.Handler(model);

            var userId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedUserId));
            var user = await this.usersServices.FindAsync(userId);
            if (user == null || user.UserId == 0)
            {
                return this.BadRequest("Sorry! We don't have a user in the system");
            }

            return this.Success(user);
        }

        /// <summary>
        /// The add user.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - User added successfully.
        /// - 409 - User already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1126:PrefixCallsCorrectly", Justification = "Reviewed. Suppression is OK here.")]
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] UserModel model, [FromHeader] LocationHeader header)
        {
            model = (UserModel)EmptyFilter.Handler(model);
            model.FullName = CoreFilter.FullName(model.FirstName, model.MiddleName, model.LastName);

            var (accountId, userId, guid) = await this.usersServices.AddFtpAsync(model);
            switch (accountId)
            {
                case -1:
                    return this.Conflict("Given email address has been exists with us.");
                case -2:
                    return this.Conflict("Given mobile number has been exists with us.");
                case 0:
                    return this.ServerError();
            }
            if (!string.IsNullOrEmpty(model.Base64ProfileImage))
            {
                var filePath = $@"{this.runningEnvironment.CurrentEnvironment}/{guid}/ProfileImage";

                try
                {
                    await this.ftpUploadHelper.CreateDirectory(filePath);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.Message);
                }
                var dbPath = $@"{model.FullName}_{DateTime.UtcNow.Ticks}.jpg";
                filePath += $@"/{dbPath}";

                try
                {
                    var uploadResponse = await this.ftpUploadHelper.UploadProfileImageAsync(model.Base64ProfileImage, filePath);
                    if (uploadResponse > 0)
                    {
                        model.ProfileImageUrl = $@"ProfileImage/{dbPath}";
                        model.ThumbnailUrl = $@"ProfileImage/{dbPath}";
                    }
                }
                catch (Exception)
                {
                    model.ProfileImageUrl = null;
                    model.ThumbnailUrl = null;
                }
            }

            if (!IsNullOrEmpty(model.ProfileImageUrl))
            {
                await this.usersServices.UpdateImageUrlsAsync(model, (Guid)guid);
            }
            try
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Users,
                    LogFrom = (short)(int)model.CreatedByRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $" {model.ModifiedByName} has added {model.FullName} user details on {DateTime.UtcNow.AddMinutes(330)}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            catch (Exception)
            {
                // ignore
            }

            var createPasswordLink = this.applicationConfiguration.CreatePasswordLink + this.aesHelper.Encode(accountId.ToString());
            return this.Success("User details has been added successfully.");
        }

        /// <summary>
        /// The update user.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - User updated successfully.
        /// - 409 - User already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] UserModel model, [FromHeader] LocationHeader header)
        {
            if (model == null)
            {
                return this.Failed("Model is null.");
            }
            model = (UserModel)EmptyFilter.Handler(model);
            model.FullName = CoreFilter.FullName(model.FirstName, model.MiddleName, model.LastName);
            var guid = await this.accountServices.FindGuidAsync(model.UserId, (Roles)model.RoleId);
            var filePath = $@"{this.runningEnvironment.CurrentEnvironment}/{guid}/ProfileImage";
            if (!string.IsNullOrEmpty(model.Base64ProfileImage))
            {
                try
                {
                    await this.ftpUploadHelper.CreateDirectory(filePath);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.Message);
                }
                var dbPath = $@"{model.FullName}_{DateTime.UtcNow.Ticks}.jpg";
                filePath += $@"/{dbPath}";

                try
                {
                    var uploadResponse = await this.ftpUploadHelper.UploadProfileImageAsync(model.Base64ProfileImage, filePath);
                    if (uploadResponse > 0)
                    {
                        var getOld = await this.usersServices.FindAsync(model.UserId);
                        if (getOld != null && !string.IsNullOrEmpty(getOld.ProfileImageUrl))
                        {
                            await this.ftpUploadHelper.DeleteFile(getOld.ProfileImageUrl);
                        }
                        model.ProfileImageUrl = $@"ProfileImage/{dbPath}";
                        model.ThumbnailUrl = $@"ProfileImage/{dbPath}";
                    }
                }
                catch (Exception)
                {
                    model.ProfileImageUrl = null;
                    model.ThumbnailUrl = null;
                }
            }

            var response = await this.usersServices.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given email address has been exists with us.");
                case -2:
                    return this.Conflict("Given mobile number has been exists with us.");
                case -3:
                    return this.Conflict("Can not update the location.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Users,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has updated {model.FullName} user details on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("User details has been updated successfully.");
        }

        /// <summary>
        /// The modify status of user.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - User status updated successfully.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] UserModel model, [FromHeader] LocationHeader header)
        {
            model = (UserModel)EmptyFilter.Handler(model);
            var response = await this.usersServices.ModifyStatusAsync(model.UserId, model.Active, Convert.ToInt32(model.ModifiedBy));
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Users,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = model.Active == true ? $" {model.ModifiedByName} has updated to Active {model.FullName} user status on {DateTime.UtcNow.AddMinutes(330)}" : $" {model.ModifiedByName} has updated to InActive {model.FullName} user status on {DateTime.UtcNow.AddMinutes(330)}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("User status has been updated successfully.");
        }

        /// <summary>
        /// The locked status async.
        /// </summary>
        /// <param name="request">
        /// The request.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPut]
        [Authorize]
        [Route("lock-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(typeof(string), 409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> LockedStatusAsync([FromBody] UserModel request, [FromHeader] LocationHeader header)
        {
            var fullName = await this.usersServices.FindFullNameByUserId(request.UserId);
            var response = await this.usersServices.LockedStatusAsync(request);

            switch (response)
            {
                case 0:
                    return this.ServerError();
                default:
                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = request.ModifiedBy,
                        LogTypeId = (int)LogTypes.Users,
                        LogFrom = (int)AccountType.Administrator,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogDescription = request.IsLocked == true ? $" {request.ModifiedByName} has updated to Locked {fullName} user status on {DateTime.UtcNow.AddMinutes(330)}" : $" {request.ModifiedByName} has updated to UnLocked {fullName} user status on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);

                    return this.Success("User has been " + (request.IsLocked ? "locked" : "Unlock") + " successfully.");
            }
        }
    }
}